import json
import xbmc
import xbmcplugin
import xbmcaddon
import xbmcgui
import os
import xbmcvfs
import uuid
from urllib.parse import parse_qsl, urlencode
import resources.lib.ttvapi as ttv
import resources.lib.util.loghelper as log
import resources.lib.util.tinycache as cache
import resources.lib.util.httphelper as http


TOAST_INFO = 0
TOAST_NOTICE = 1
TOAST_ERROR = 2


def toast(title, message, toast_type):
    toast_icon = ''
    if toast_type == TOAST_ERROR:
        toast_icon = xbmcgui.NOTIFICATION_ERROR
    elif toast_type == TOAST_INFO:
        toast_icon = xbmcgui.NOTIFICATION_INFO
    elif toast_type == TOAST_NOTICE:
        toast_icon = xbmcgui.NOTIFICATION_INFO
    xbmcgui.Dialog().notification(title, message, toast_icon)


class TTVStrings:
    """Collection of localized strings"""
    def __init__(self, addon):
        self.SECTION_FILMS = addon.getLocalizedString(30023)
        self.SECTION_SERIES = addon.getLocalizedString(30024)
        self.SECTION_ANIMATION = addon.getLocalizedString(30025)
        self.SECTION_POFILE = addon.getLocalizedString(30026)
        self.ACTION_LOGIN = addon.getLocalizedString(30000)
        self.ACTION_LOGOUT = addon.getLocalizedString(30004)
        self.ACTION_SEARCH = addon.getLocalizedString(30005)
        self.ACTION_SEARCH_CLEAR = addon.getLocalizedString(30049)
        self.ACTION_NEW_SEARCH = addon.getLocalizedString(30006)
        self.MEDIA_LIST_NEXT_PAGE = addon.getLocalizedString(30010)
        self.MEDIA_LIST_PREV_PAGE = addon.getLocalizedString(30011)
        self.MEDIA_VIEW_POPULAR = addon.getLocalizedString(30027)
        self.MEDIA_VIEW_KP_RATING = addon.getLocalizedString(30028)
        self.MEDIA_VIEW_IMDB_RATING = addon.getLocalizedString(30029)
        self.MEDIA_VIEW_AIR_DATE = addon.getLocalizedString(30030)
        self.MEDIA_VIEW_DATE_ADDED = addon.getLocalizedString(30031)
        self.MEDIA_VIEW_GENRE = addon.getLocalizedString(30032)
        self.MEDIA_ALL_GENRES = addon.getLocalizedString(30046)
        self.MEDIA_FILE_SELECT = addon.getLocalizedString(30047)
        self.MEDIA_BACK_TO_MAIN = addon.getLocalizedString(30048)
        self.PROFILE_ACCOUNT = addon.getLocalizedString(30002)
        self.ACCOUNT_LOGON_FAILED = addon.getLocalizedString(30033)
        self.ACCOUNT_NOT_LOGGED_IN = addon.getLocalizedString(30034)
        self.ACCOUNT_BALANCE = addon.getLocalizedString(30037)
        self.ACCOUNT_STATUS = addon.getLocalizedString(30038)
        self.ACCOUNT_VIP = addon.getLocalizedString(30039)
        self.ACCOUNT_LIFETIME_VIP = addon.getLocalizedString(30040)
        self.ACCOUNT_REGULAR = addon.getLocalizedString(30041)
        self.CONFIRM = addon.getLocalizedString(30035)
        self.CONFIRM_LOGOUT = addon.getLocalizedString(30036)
        self.ERROR = addon.getLocalizedString(30012)
        self.INFO = addon.getLocalizedString(30013)
        self.NOTICE = addon.getLocalizedString(30014)
        self.ERROR_REQUEST_FAILED = addon.getLocalizedString(30045)


class TTVSearchHistory:
    def __init__(self, addon, size=10):
        self._addon = addon
        self._path = os.path.join(
            xbmcvfs.translatePath(self._addon.getAddonInfo('profile')),
            'searches.json')
        self._history = []
        self.max_size = size
        self.load()

    @property
    def history(self):
        return self._history.copy()

    def append(self, item):
        if item in self._history:
            return

        if len(self._history) > self.max_size - 1:
            self._history.pop(0)
        self._history.append(item)
        self.save()

    def load(self):
        if not xbmcvfs.exists(self._path):
            self.save()
        else:
            try:
                with xbmcvfs.File(self._path) as file:
                    self._history = json.load(file)
            except Exception as e:
                log.write_line('Could not load search history! ( %s )' % e, log.LOG_LEVEL_ERROR)

    def clear(self):
        self._history.clear()
        self.save()

    def save(self):
        try:
            with xbmcvfs.File(self._path, 'w') as file:
                json.dump(self._history, file)
        except Exception as e:
            log.write_line('Could not save search history! ( %s )' % e, log.LOG_LEVEL_ERROR)


class TTVSettings:
    def __init__(self, addon):
        self._addon = addon
        self._settings = {'username': '',
                          'password': '',
                          'list_order': '',
                          'items_per_page': '10',
                          'session_id': '',
                          'rating_source': '',
                          'uuid': ''
                          }
        self.reload()

    def __getitem__(self, item):
        return self._settings[item]

    def __setitem__(self, key, value):
        self._settings[key] = value
        self._addon.setSetting(key, value)

    def reload(self):
        """Reload settings"""
        for key in self._settings:
            self._settings[key] = self._addon.getSetting(key)

    def save(self):
        """Save settings"""
        for key in self._settings:
            self._addon.setSetting(key, self._settings[key])


class TTVPlugin:
    def __init__(self, args):
        self.id = 'plugin.video.ttv-video-client'
        self.addon = xbmcaddon.Addon(self.id)
        self.strings = TTVStrings(self.addon)
        self._settings = TTVSettings(self.addon)
        self._args = args
        self._path = args[2][1:]
        self._handle = int(args[1])
        self._url = args[0]

        if not self._settings['uuid']:
            self._settings['uuid'] = str(uuid.uuid1()).replace('-', '')
        self.ttv_api = ttv.TTVApi(self._settings['username'],
                                  self._settings['password'],
                                  self._settings['uuid'],
                                  self._settings['session_id'])
        self._main_win = xbmcgui.Window(10000)
        if self.ttv_api.logged_in:
            self._cache = cache.TinyCache(self.id)
            http.CACHE = self._cache
            http.CACHE_TTL_DEFAULT = 300
            http.NO_CACHE_URLS.append('/v3/auth.php')


    def __del__(self):
        if self.ttv_api is None:
            return
        # save session if refreshed by api on exit
        if self._settings['session_id'] != self.ttv_api.session:
            self._settings['session_id'] = self.ttv_api.session

    def _get_url(self, **params):
        return '{}?{}'.format(self._url, urlencode(params))

    def _refresh_listing(self):
        xbmc.executebuiltin('Container.Refresh(%s)' % self._url, False)

    def _get_film_play_info(self, film_id):
        film_info = self.ttv_api.get_film_alt_info(film_id)
        if self.ttv_api.result_ok(film_info):
            if film_info.get('data') is not None:
                # insert film_id for sanity check
                # and reset all Nones with empty strings
                info = film_info['data']
                info['film_id'] = film_id
                if info.get('name') is None:
                    info['name'] = ''
                if info.get('description') is None:
                    info['description'] = ''
                if info.get('poster') is None:
                    info['poster'] = ''
                if info.get('year') is None:
                    info['year'] = 0
                # save info as kodi home window property
                self._main_win.setProperty('{}-{}'.format(self.id, 'film_info'),
                                           json.dumps(info))
        else:
            self._main_win.setProperty('{}-{}'.format(self.id, 'film_info'), '')

    def _restore_film_play_info(self, film_id):
        info = self._main_win.getProperty('{}-{}'.format(self.id, 'film_info'))
        if info:
            data = json.loads(info)
            if data.get('film_id') == film_id:
                return data
        return None

    def _add_action_item(self, label, icon='', is_folder=True, **param):
        list_item = xbmcgui.ListItem(label)
        list_item.setArt({'icon': icon})
        xbmcplugin.addDirectoryItem(self._handle,
                                    self._get_url(**param),
                                    list_item,
                                    is_folder)

    def _add_status_kv_item(self, key, value):
        list_item = xbmcgui.ListItem('{0}: {1}'.format(key, value))
        xbmcplugin.addDirectoryItem(self._handle, self._url, list_item, False)

    def _add_movie_item(self, data, media_type=ttv.FILM_TYPE_FILM):
        name = data.get('name', '')
        film_id = int(data.get('id', 0))
        if not name or not film_id:
            # broken or corrupt item with empty name or id
            return

        # any key can be defined and set to None leading to plugin crash
        # double check everything, just in case
        if data.get('poster') is not None:
            poster = data['poster']
        else:
            poster = ''
        if data.get('description') is not None:
            description = data['description']
        else:
            description = ''
        if data.get(self._settings['rating_source']) is not None:
            rating = data[self._settings['rating_source']]
        else:
            rating = '0.0'
        if data.get('year') is not None:
            year = data['year']
        else:
            year = 0
        if data.get('country') is not None:
            country = data['country']
        else:
            country = ''

        list_item = xbmcgui.ListItem(name)
        list_item.setArt({'icon': poster,
                          'thumb': poster,
                          'poster': poster})
        list_item.setInfo('video', {'plot': description + '...',
                                    'rating': float(rating),
                                    'year': year,
                                    'country': country})
        list_item.setProperty('IsPlayable', 'true')
        xbmcplugin.addDirectoryItem(self._handle,
                                    self._get_url(media_type=media_type,
                                                  action='play',
                                                  film_id=film_id),
                                    list_item,
                                    False)

    def _list_films(self, param):
        if not self.ttv_api.logged_in:
            toast(self.strings.ERROR, self.strings.ACCOUNT_NOT_LOGGED_IN, TOAST_ERROR)
            return

        media_type = param.get('media_type', ttv.FILM_TYPE_FILM)
        genre = param.get('genre_id', 0)
        order = self._settings['list_order']
        skip = int(param.get('skip', 0))
        limit = int(self._settings['items_per_page'])

        if media_type == ttv.FILM_TYPE_SERIES:
            xbmcplugin.setContent(self._handle, 'tvshows')
        else:
            xbmcplugin.setContent(self._handle, 'movies')

        res = self.ttv_api.get_film_alt_list(media_type,
                                             genre,
                                             self._settings['list_order'],
                                             skip,
                                             limit)
        if not self.ttv_api.result_ok(res):
            log.write_line('Could not get media list! ( %s )' % repr(res), log.LOG_LEVEL_ERROR)
            return

        for item in res['data']:
            self._add_movie_item(item, media_type)

        if skip != 0:
            list_prev = xbmcgui.ListItem(self.strings.MEDIA_LIST_PREV_PAGE)
            xbmcplugin.addDirectoryItem(self._handle, self._get_url(media_type=media_type,
                                                                    action='list_media',
                                                                    order=order,
                                                                    skip=skip - limit,
                                                                    limit=limit,
                                                                    genre_id=genre),
                                        list_prev,
                                        True)

        if len(res['data']) == limit:
            list_next = xbmcgui.ListItem(self.strings.MEDIA_LIST_NEXT_PAGE)
            xbmcplugin.addDirectoryItem(self._handle, self._get_url(media_type=media_type,
                                                                    action='list_media',
                                                                    order=order,
                                                                    skip=skip + limit,
                                                                    limit=limit,
                                                                    genre_id=genre),
                                        list_next,
                                        True)
        xbmcplugin.endOfDirectory(self._handle, updateListing=True)

    def _genre_select(self, param):
        if not self.ttv_api.logged_in:
            toast(self.strings.ERROR, self.strings.ACCOUNT_NOT_LOGGED_IN, TOAST_ERROR)
            return

        genres = self.ttv_api.get_film_alt_genres()
        if not self.ttv_api.result_ok(genres):
            toast(self.strings.ERROR, self.strings.ERROR_REQUEST_FAILED, TOAST_ERROR)
            return

        video_type = param['media_type']
        self._add_action_item(self.strings.MEDIA_ALL_GENRES, '',
                              is_folder=True,
                              action='list_media',
                              media_type=video_type,
                              genre_id=0)

        for cat in genres['data']:
            if cat['type'] == param['media_type']:
                for genre in cat['genres']:
                    # yes technically every key can exist and be set to none
                    # in server response after json reply is deserialized
                    # or set to none by the server
                    if genre.get('name', None) is not None:
                        genre_name = genre['name']
                    else:
                        genre_name = ''
                    genre_id = 0
                    if 'query' in genre:
                        if genre['query'].get('genre_id', 0) is not None:
                            genre_id = genre['query'].get('genre_id', 0)
                    if genre_id == 0:
                        continue

                    self._add_action_item(genre_name, '',
                                          is_folder=True,
                                          action='list_media',
                                          media_type=video_type,
                                          genre_id=genre_id)
        xbmcplugin.endOfDirectory(self._handle)

    def _search_menu(self, param):
        if not self.ttv_api.logged_in:
            toast(self.strings.ERROR, self.strings.ACCOUNT_NOT_LOGGED_IN, TOAST_ERROR)
            return

        self._add_action_item(self.strings.ACTION_NEW_SEARCH, 'DefaultAddonsSearch.png',
                              is_folder=True,
                              action='new_search')

        history = TTVSearchHistory(self.addon).history
        if len(history) > 0:
            self._add_action_item(self.strings.ACTION_SEARCH_CLEAR, 'DefaultAddonNone.png',
                                  is_folder=False,
                                  action='clear_searches')

        for item in history:
            self._add_action_item(item, 'DefaultFile.png',
                                  is_folder=True,
                                  action='new_search',
                                  queue=item)

        xbmcplugin.endOfDirectory(self._handle)

    def _main_menu(self):
        self._add_action_item(self.strings.SECTION_FILMS, is_folder=True,
                              action='genre_select',
                              media_type=ttv.FILM_TYPE_FILM)
        self._add_action_item(self.strings.SECTION_ANIMATION, is_folder=True,
                              action='genre_select',
                              media_type=ttv.FILM_TYPE_ANIMATION)
        self._add_action_item(self.strings.SECTION_SERIES, is_folder=True,
                              action='genre_select',
                              media_type=ttv.FILM_TYPE_SERIES)
        self._add_action_item(self.strings.ACTION_SEARCH, 'DefaultAddonsSearch.png',
                              True,
                              action='search')

        if not self.ttv_api.logged_in:
            self._add_action_item(self.strings.ACTION_LOGIN, 'OverlayLocked.png',
                                  False,
                                  action='login')
        else:
            self._add_action_item(self.strings.SECTION_POFILE, 'DefaultUser.png',
                                  True,
                                  action='view_profile')
        xbmcplugin.endOfDirectory(self._handle)

    def _list_user_info(self):
        res = self.ttv_api.get_user_info()
        if not self.ttv_api.result_ok(res):
            toast(self.strings.ERROR, self.strings.ACCOUNT_NOT_LOGGED_IN, TOAST_ERROR)
            return

        account = res.get('email', '')
        balance = res.get('balance', 0)
        vip_status = res.get('vip_status', False)
        vip_life = res.get('vip_life', False)

        status = self.strings.ACCOUNT_REGULAR
        if vip_status:
            status = self.strings.ACCOUNT_VIP
        if vip_life:
            status = self.strings.ACCOUNT_LIFETIME_VIP

        self._add_status_kv_item(self.strings.PROFILE_ACCOUNT, account)
        self._add_status_kv_item(self.strings.ACCOUNT_BALANCE, balance)
        self._add_status_kv_item(self.strings.ACCOUNT_STATUS, status)

        self._add_action_item(self.strings.ACTION_LOGOUT, 'DefaultAddonNone.png',
                              is_folder=False,
                              action='logout')

        xbmcplugin.endOfDirectory(self._handle)

    def _select_episode(self, param):
        xbmcplugin.setContent(self._handle, 'episodes')
        film_http = self.ttv_api.get_film_alt_http(param['film_id'])
        if not self.ttv_api.result_ok(film_http):
            log.write_line('Could not request film data! ( %s )' % film_http, log.LOG_LEVEL_ERROR)
            toast(self.strings.ERROR, 'Missing film url', TOAST_ERROR)
            return

        film_data = film_http.get('data')
        if film_data is None or len(film_data) == 0:
            log.write_line('Could not get film data! ( %s )' % film_http, log.LOG_LEVEL_ERROR)
            toast(self.strings.ERROR, 'Missing stream', TOAST_ERROR)
            return

        for episode in film_data:
            list_item = xbmcgui.ListItem(episode['name'])
            list_item.setProperty('IsPlayable', 'true')
            xbmcplugin.addDirectoryItem(self._handle,
                                        self._get_url(media_type=param['media_type'],
                                                      action='play',
                                                      film_id=param['film_id'],
                                                      video_url=episode['url'],
                                                      name=episode['name']),
                                        list_item,
                                        False)
        xbmcplugin.endOfDirectory(self._handle)

    def _play(self, param):
        if param.get('film_id') is not None:
            film_id = param['film_id']
        else:
            return

        media_type = ttv.FILM_TYPE_FILM
        if param.get('media_type'):
            media_type = param['media_type']

        play_url = None
        if param.get('video_url') is not None:
            play_url = param['video_url']
        else:
            film_http = self.ttv_api.get_film_alt_http(film_id)
            if not self.ttv_api.result_ok(film_http):
                log.write_line('Could not request film data! ( %s )' % film_http, log.LOG_LEVEL_ERROR)
                toast(self.strings.ERROR, 'Missing film url', TOAST_ERROR)
                return

            film_data = film_http.get('data')
            if film_data is None or len(film_data) == 0:
                log.write_line('Could not get film data! ( %s )' % film_http, log.LOG_LEVEL_ERROR)
                toast(self.strings.ERROR, 'Missing stream', TOAST_ERROR)
                return

            self._get_film_play_info(film_id)

            if len(film_data) == 1:
                play_url = film_data[0]['url']
            else:
                xbmc.executebuiltin('Container.Refresh(%s)' % self._get_url(action='select_episode',
                                                                            media_type=media_type,
                                                                            film_id=film_id))
        if play_url:
            play_item = xbmcgui.ListItem(path=play_url)
            info = self._restore_film_play_info(film_id)
            episode_name = ''
            if param.get('name'):
                episode_name = ' [ %s ]' % param['name']
            if info is not None:
                play_item.setInfo('video', {
                    'title': info['name'] + episode_name,
                    'originaltitle': info['name'],
                    'plot': info['description'],
                    'year': info['year']
                })
                play_item.setArt({'poster': info['poster'], 'thumb': info['poster']})

            xbmcplugin.setResolvedUrl(self._handle, True, listitem=play_item)

    def _logout(self):
        res = xbmcgui.Dialog().yesno(self.strings.CONFIRM, self.strings.CONFIRM_LOGOUT)
        if not res:
            return

        self._settings['username'] = '@'
        self._settings['password'] = ''
        self._settings['session_id'] = ''
        self.ttv_api = None

        self._refresh_listing()

    def _login(self):
        xbmc.executebuiltin('Addon.OpenSettings(%s)' % self.id, True)
        self._settings.reload()
        self.ttv_api = ttv.TTVApi(self._settings['username'],
                                  self._settings['password'],
                                  'toast',
                                  self._settings['session_id'])
        if not self.ttv_api.result_ok(self.ttv_api.auth()):
            toast(self.strings.ERROR, self.strings.ACCOUNT_LOGON_FAILED, TOAST_ERROR)
        self._refresh_listing()

    def _searches_clear(self):
        searches = TTVSearchHistory(self.addon)
        searches.clear()
        xbmc.executebuiltin('Container.Refresh(%s)' % self._get_url(action='search'))

    def _search(self, params):
        queue = params.get('queue', '')
        if not queue:
            queue = xbmcgui.Dialog().input(self.strings.ACTION_SEARCH, type=xbmcgui.INPUT_ALPHANUM)
            xbmc.executebuiltin('Container.Refresh(%s)' % self._get_url(action='new_search', queue=queue))
            xbmcplugin.endOfDirectory(self._handle)
            return

        if queue:
            TTVSearchHistory(self.addon).append(queue)
            res = self.ttv_api.film_alt_search(queue)
            if self.ttv_api.result_ok(res):
                if res.get('data') is not None:
                    for item in res['data']:
                        self._add_movie_item(item)
        xbmcplugin.endOfDirectory(self._handle)

    def handle(self):
        params = dict(parse_qsl(self._path))
        if not params:
            self._main_menu()
        elif params['action'] == 'play':
            self._play(params)
        elif params['action'] == 'list_media':
            self._list_films(params)
        elif params['action'] == 'select_episode':
            self._select_episode(params)
        elif params['action'] == 'genre_select':
            self._genre_select(params)
        elif params['action'] == 'search':
            self._search_menu(params)
        elif params['action'] == 'new_search':
            self._search(params)
        elif params['action'] == 'clear_searches':
            self._searches_clear()
        elif params['action'] == 'view_profile':
            self._list_user_info()
        elif params['action'] == 'logout':
            self._logout()
        elif params['action'] == 'login':
            self._login()
        else:
            log.write_line('bad route: %s' % repr(params), log.LOG_LEVEL_ERROR)
            toast(self.strings.ERROR, 'BAD ROUTE', TOAST_ERROR)
